<?php
defined('BASEPATH') or exit('No direct script access allowed');

if (!function_exists('adjust_brightness')) {
    /**
     * Adjust brightness of a hex color.
     *
     * @param string $hex Hex color code (e.g. #aabbcc)
     * @param int $steps Amount to adjust brightness (-255 to 255)
     * @return string Adjusted hex color code
     */
    function adjust_brightness($hex, $steps)
    {
        // Steps should be between -255 and 255. Negative = darker, positive = lighter
        $steps = max(-255, min(255, $steps));

        // Remove '#' if present
        $hex = str_replace('#', '', $hex);

        // If shorthand notation, convert to full form, e.g. 'abc' -> 'aabbcc'
        if (strlen($hex) == 3) {
            $hex = $hex[0] . $hex[0] .
                $hex[1] . $hex[1] .
                $hex[2] . $hex[2];
        }

        // Split into RGB components
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));

        // Adjust each component
        $r = max(0, min(255, $r + $steps));
        $g = max(0, min(255, $g + $steps));
        $b = max(0, min(255, $b + $steps));

        // Convert back to hex and return
        return sprintf("#%02x%02x%02x", $r, $g, $b);
    }
}

if (!function_exists('apply_theme')) {
    function apply_theme()
    {
        $CI =& get_instance();
        if (!isset($CI->Settings_model)) {
            $CI->load->model('Settings_model');
        }

        $settings = $CI->Settings_model->get_settings();
        if (!$settings) {
            return '';
        }

        $primary_color = $settings->primary_color;
        $secondary_color = $settings->secondary_color;
        $text_color = $settings->text_color ?? '#000000';

        $theme_css = '<style>';
        $theme_css .= ":root {
            --primary-color: {$primary_color};
            --secondary-color: {$secondary_color};
            --text-color: {$text_color};
            --primary-hover: " . adjust_brightness($primary_color, -20) . ";
            --secondary-hover: " . adjust_brightness($secondary_color, -20) . ";
        }";

        $background_style = $CI->Settings_model->get_background_style();
        $theme_css .= "body { {$background_style} }";

        // Overlay for image backgrounds
        if ($settings->background_type === 'image' && $settings->background_overlay) {
            $theme_css .= "
                body::before {
                    content: '';
                    position: fixed;
                    top: 0;
                    left: 0;
                    width: 100%;
                    height: 100%;
                    background: linear-gradient(135deg, rgba(0,0,0,0.4) 0%, rgba(0,0,0,0.6) 100%);
                    z-index: -1;
                }
            ";
        }

        // Global text color for all elements
        $theme_css .= "
            body, html, * {
                color: var(--text-color) !important;
            }
            
            /* Override for specific elements */
            h1, h2, h3, h4, h5, h6, p, span, div, label, input, textarea, select {
                color: var(--text-color) !important;
            }
            
            /* Keep link colors */
            a {
                color: var(--primary-color) !important;
            }
            a:hover {
                color: var(--primary-hover) !important;
            }
            
            /* Tailwind CSS color overrides */
            .bg-primary { background-color: var(--primary-color) !important; }
            .bg-secondary { background-color: var(--secondary-color) !important; }
            .text-primary { color: var(--primary-color) !important; }
            .text-secondary { color: var(--secondary-color) !important; }
            .text-default { color: var(--text-color) !important; }
            .border-primary { border-color: var(--primary-color) !important; }
            .border-secondary { border-color: var(--secondary-color) !important; }
            .btn-primary { background-color: var(--primary-color) !important; border-color: var(--primary-color) !important; }
            .btn-secondary { background-color: var(--secondary-color) !important; border-color: var(--secondary-color) !important; }
            .btn-primary:hover { background-color: var(--primary-hover) !important; border-color: var(--primary-hover) !important; }
            .btn-secondary:hover { background-color: var(--secondary-hover) !important; border-color: var(--secondary-hover) !important; }
        ";

        $theme_css .= '</style>';

        return $theme_css;
    }
}

if (!function_exists('get_theme_colors')) {
    function get_theme_colors()
    {
        $CI =& get_instance();
        $CI->load->model('Settings_model');
        return $CI->Settings_model->get_theme_colors();
    }
}
?>