<?php
defined('BASEPATH') or exit('No direct script access allowed');

/**
 * Generate embed HTML for supported video links.
 * Supports YouTube, Facebook, Telegram, Google Drive, Google Live Streaming.
 *
 * @param string $url Video URL or embed link
 * @param int $width Width of the embed iframe
 * @param int $height Height of the embed iframe
 * @return string Embed HTML iframe or empty string if unsupported
 */
function get_video_embed_html($url, $width = 560, $height = 315)
{
    if (empty($url)) {
        return '';
    }

    // YouTube
    if (preg_match('/(?:youtube\.com\/watch\?v=|youtu\.be\/)([^\&\?\/]+)/', $url, $matches)) {
        $video_id = $matches[1];
        return '<iframe width="' . $width . '" height="' . $height . '" src="https://www.youtube.com/embed/' . $video_id . '" frameborder="0" allowfullscreen></iframe>';
    }

    // Facebook public video post
    if (strpos($url, 'facebook.com') !== false) {
        $encoded_url = urlencode($url);
        return '<iframe src="https://www.facebook.com/plugins/video.php?href=' . $encoded_url . '&show_text=0" width="' . $width . '" height="' . $height . '" style="border:none;overflow:hidden" scrolling="no" frameborder="0" allowfullscreen="true" allow="autoplay; clipboard-write; encrypted-media; picture-in-picture; web-share"></iframe>';
    }

    // Telegram direct video link (assumed to be a direct video file link)
    if (strpos($url, 'telegram.org') !== false || strpos($url, 't.me') !== false) {
        // Check if URL ends with a video file extension
        if (preg_match('/\.(mp4|webm|ogg)$/i', $url)) {
            return '<video width="' . $width . '" height="' . $height . '" controls><source src="' . htmlspecialchars($url) . '" type="video/mp4">Your browser does not support the video tag.</video>';
        } else {
            // Unsupported Telegram link format
            return '<p style="color: red; font-weight: bold;">Unsupported Telegram video link format. Only direct video file URLs (e.g., ending with .mp4) are supported for Telegram videos. Telegram message or channel links cannot be played directly. Please provide a direct video file URL from Telegram or upload the video file.</p>';
        }
    }

    // Google Drive embed/share link
    if (strpos($url, 'drive.google.com') !== false) {
        // Extract file ID from various Google Drive URL formats
        if (preg_match('/\/file\/d\/([a-zA-Z0-9_-]+)/', $url, $matches) ||
            preg_match('/id=([a-zA-Z0-9_-]+)/', $url, $matches)) {
            $file_id = $matches[1];
            $embed_url = 'https://drive.google.com/file/d/' . $file_id . '/preview';
            return '<iframe src="' . $embed_url . '" width="' . $width . '" height="' . $height . '" allow="autoplay" frameborder="0" allowfullscreen></iframe>';
        }
    }

    // Google Live Streaming embed link (e.g., LiveStreaming.aspx)
    if (strpos($url, 'LiveStreaming.aspx') !== false) {
        return '<iframe src="' . htmlspecialchars($url) . '" width="' . $width . '" height="' . $height . '" frameborder="0" allowfullscreen allow="autoplay; encrypted-media"></iframe>';
    }

    // Unsupported URL
    return '';
}
