<?php
defined('BASEPATH') or exit('No direct script access allowed');

/**
 * @property CI_Session $session
 * @property CI_Form_validation $form_validation
 * @property CI_Input $input
 * @property Video_model $Video_model
 * @property User_model $User_model
 * @property Live_model $Live_model
 * @property CI_Upload $upload
 */
class User extends MY_Controller
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model(['Video_model', 'User_model', 'Live_model']);
        $this->load->library(['session', 'upload', 'form_validation']);
        $this->load->helper(['form', 'url', 'theme_helper']);
    }

    private function _is_logged_in()
    {
        if (!$this->session->userdata('user_id')) {
            redirect('login');
        }
    }

    public function dashboard()
    {
        $this->_is_logged_in();
        $user_id = $this->session->userdata('user_id');
        $data['videos'] = $this->Video_model->get_videos_by_user($user_id);

        // Get admin videos with video_type information
        $this->db->select('videos.*, users.username as uploader');
        $this->db->from('videos');
        $this->db->join('users', 'videos.user_id = users.id');
        $this->db->where_in('users.role', ['admin', 'superadmin', 'developer']);
        $this->db->order_by('videos.created_at', 'DESC');
        $data['admin_videos'] = $this->db->get()->result();

        // Debug output for admin videos count
        log_message('debug', 'Admin videos count: ' . count($data['admin_videos']));

        $this->load->view('templates/header');
        $this->load->view('templates/sidebar');
        $this->load->view('user/dashboard', $data);
        $this->load->view('templates/footer');
    }

    public function myvideo()
    {
        $this->_is_logged_in();
        $user_id = $this->session->userdata('user_id');
        $data['videos'] = $this->Video_model->get_videos_by_user($user_id);

        $this->load->view('templates/header');
        $this->load->view('templates/sidebar');
        $this->load->view('user/myvideo', $data);
        $this->load->view('templates/footer');
    }

    public function upload_video()
    {
        $this->_is_logged_in();

        if ($this->input->method() === 'post') {
            $link = $this->input->post('link', true);
            $title = $this->input->post('title', true);
            $description = $this->input->post('description', true);
            $file = $_FILES['video_file'];
            $thumbnail_file = $_FILES['thumbnail_file'];

            if (empty($link) && empty($file['name'])) {
                $this->session->set_flashdata('error', 'Harap masukkan link atau upload video.');
                redirect('upload-video');
                return;
            }

            // Validate Facebook video URL if link contains facebook.com
            if (!empty($link) && strpos($link, 'facebook.com') !== false) {
                // Reject profile.php?id= URLs explicitly
                if (strpos($link, 'profile.php?id=') !== false) {
                    $this->session->set_flashdata('error', 'URL Facebook tidak valid. Harap masukkan URL video Facebook yang benar, bukan profil.');
                    redirect('upload-video');
                    return;
                }
                // Facebook video post URL pattern example:
                // https://www.facebook.com/{page_or_user}/videos/{video_id}/
                $pattern = '/^https?:\/\/(www\.)?facebook\.com\/[^\/]+\/videos\/[0-9]+\/?/';
                if (!preg_match($pattern, $link)) {
                    $this->session->set_flashdata('error', 'URL Facebook tidak valid. Harap masukkan URL video Facebook yang benar.');
                    redirect('upload-video');
                    return;
                }
            }

            $source = '';
            $source_type = '';
            $thumbnail_path = '';

            if (!empty($link)) {
                $source_type = 'link';
                $source = $link;
            } elseif (!empty($file['name'])) {
                $source_type = 'file';
                $config['upload_path'] = './uploads/videos/';
                $config['allowed_types'] = 'mp4|mkv|mov|avi';
                $config['max_size'] = 200000; // 200MB

                if (!is_dir($config['upload_path'])) {
                    mkdir($config['upload_path'], 0777, true);
                }

                $this->upload->initialize($config);

                if (!$this->upload->do_upload('video_file')) {
                    $this->session->set_flashdata('error', $this->upload->display_errors('', ''));
                    redirect('upload-video');
                    return;
                }

                $upload_data = $this->upload->data();
                $source = 'uploads/videos/' . $upload_data['file_name'];
            }

            // Handle thumbnail upload
            if (!empty($thumbnail_file['name'])) {
                $thumb_config['upload_path'] = './uploads/thumbnails/';
                $thumb_config['allowed_types'] = 'jpg|jpeg|png|gif';
                $thumb_config['max_size'] = 5000; // 5MB

                if (!is_dir($thumb_config['upload_path'])) {
                    mkdir($thumb_config['upload_path'], 0777, true);
                }

                $this->upload->initialize($thumb_config);

                if (!$this->upload->do_upload('thumbnail_file')) {
                    $this->session->set_flashdata('error', $this->upload->display_errors('', ''));
                    redirect('upload-video');
                    return;
                }

                $thumb_data = $this->upload->data();
                $thumbnail_path = 'uploads/thumbnails/' . $thumb_data['file_name'];
            }

            $data = [
                'title' => $title,
                'description' => $description,
                'user_id' => $this->session->userdata('user_id'),
                'source_type' => $source_type,
                'source' => $source,
                'thumbnail' => $thumbnail_path,
                'created_at' => date('Y-m-d H:i:s'),
            ];

            $this->Video_model->upload_video($data);
            $this->session->set_flashdata('success', 'Video berhasil diupload.');
            redirect('dashboard');
        } else {
            $this->load->view('templates/header');
            $this->load->view('templates/sidebar');
            $this->load->view('user/upload_video');
            $this->load->view('templates/footer');
        }
    }

    public function live_video()
    {
        $this->_is_logged_in();

        // Get live videos from videos table where video_type = 'live'
        $this->db->select('videos.*, users.username as streamer_name');
        $this->db->from('videos');
        $this->db->join('users', 'videos.user_id = users.id');
        $this->db->where('videos.video_type', 'live');
        $this->db->order_by('videos.created_at', 'DESC');
        $data['live_streams'] = $this->db->get()->result();

        $this->load->view('templates/header');
        $this->load->view('templates/sidebar');
        $this->load->view('user/live_video', $data);
        $this->load->view('templates/footer');
    }

    public function watch_video($video_id)
    {
        $this->_is_logged_in();

        $data['video'] = $this->Video_model->get_video_by_id($video_id);

        if (!$data['video']) {
            $this->session->set_flashdata('error', 'Video tidak ditemukan atau sudah dihapus.');
            redirect('dashboard');
            return;
        }

        // Load Comment_model
        $this->load->model('Comment_model');
        $data['comments'] = $this->Comment_model->get_comments_by_video($video_id);

        // Re-enable TelegramService extraction with fallback
        $this->load->library('TelegramService');
        $telegramService = new TelegramService();

        if ($data['video']->source_type === 'link' && (strpos($data['video']->source, 'telegram.org') !== false || strpos($data['video']->source, 't.me') !== false)) {
            $directUrl = $telegramService->extractDirectVideoUrl($data['video']->source);
            if ($directUrl) {
                $data['video']->source = $directUrl;
            }
        }

        $this->load->view('templates/header');
        $this->load->view('user/watch_video', $data);
        $this->load->view('templates/footer');
    }

    public function post_comment()
    {
        $this->_is_logged_in();

        $video_id = $this->input->post('video_id', true);
        $comment_text = $this->input->post('comment_text', true);
        $user_id = $this->session->userdata('user_id');

        if (empty($comment_text) || empty($video_id)) {
            $this->session->set_flashdata('error', 'Comment cannot be empty.');
            redirect('user/watch_video/' . $video_id);
            return;
        }

        $this->load->model('Comment_model');

        $data = [
            'video_id' => $video_id,
            'user_id' => $user_id,
            'comment_text' => $comment_text,
            'created_at' => date('Y-m-d H:i:s'),
        ];

        $this->Comment_model->add_comment($data);

        $this->session->set_flashdata('success', 'Comment added successfully.');
        redirect('user/watch_video/' . $video_id);
    }

    public function profile()
    {
        $this->_is_logged_in();

        $user_id = $this->session->userdata('user_id');
        $data['user'] = $this->User_model->get_user_by_id($user_id);

        $this->load->view('templates/header');
        $this->load->view('templates/sidebar');
        $this->load->view('user/profile', $data);
        $this->load->view('templates/footer');
    }

    public function edit_profile()
    {
        $this->_is_logged_in();

        $user_id = $this->session->userdata('user_id');

        $this->form_validation->set_rules('name', 'Name', 'required|min_length[3]');
        $this->form_validation->set_rules('email', 'Email', 'required|valid_email');
        $this->form_validation->set_rules('password', 'Password', 'min_length[6]');

        if ($this->form_validation->run() === FALSE) {
            $this->profile();
            return;
        }

        $data = [
            'name' => $this->input->post('name', true),
            'email' => $this->input->post('email', true),
            'profile_frame_style' => $this->input->post('profile_frame_style', true),
            'self_description' => $this->input->post('self_description', true),
        ];

        $password = $this->input->post('password', true);
        if (!empty($password)) {
            $data['password'] = password_hash($password, PASSWORD_DEFAULT);
        }

        // Handle profile photo upload
        if (!empty($_FILES['profile_photo']['name'])) {
            $config['upload_path'] = './assets/sbadmin2/uploads/';
            $config['allowed_types'] = 'jpg|jpeg|png|gif';
            $config['max_size'] = 2048; // 2MB
            $config['file_name'] = 'profile_' . $user_id . '_' . time();

            if (!is_dir($config['upload_path'])) {
                mkdir($config['upload_path'], 0777, true);
            }

            $this->upload->initialize($config);

            if (!$this->upload->do_upload('profile_photo')) {
                $this->session->set_flashdata('error', $this->upload->display_errors('', ''));
                redirect('myprofile');
                return;
            }

            $upload_data = $this->upload->data();
            $data['profile_photo'] = $upload_data['file_name'];
        }

        $this->User_model->update_user($user_id, $data);
        $this->session->set_flashdata('success', 'Profil berhasil diperbarui.');
        redirect('myprofile');
    }

    public function mobile_simple_video($video_id)
    {
        $this->_is_logged_in();

        $data['video'] = $this->Video_model->get_video_by_id($video_id);

        if (!$data['video']) {
            show_404();
        }

        $this->load->view('templates/header');
        $this->load->view('user/mobile_simple_video', $data);
        $this->load->view('templates/footer');
    }

    public function delete_account()
    {
        $this->_is_logged_in();

        $user_id = $this->session->userdata('user_id');

        // Delete user account
        $this->User_model->delete_user($user_id);

        // Destroy session and redirect to login
        $this->session->sess_destroy();
        $this->session->set_flashdata('success', 'Akun Anda telah dihapus.');
        redirect('auth/login');
    }

    public function forgot_password()
    {
        // Removed login check to allow access without login

        $data['title'] = 'Lupa Password - Reset Password';
        $data['user'] = null;

        $this->form_validation->set_rules('email', 'Email', 'required|valid_email');

        if ($this->form_validation->run() === FALSE) {
            $this->load->view('templates/header', $data);
            $this->load->view('user/forgot_password', $data);
            $this->load->view('templates/footer');
        } else {
            $email = $this->input->post('email');

            // Generate reset token
            $token = $this->User_model->generate_reset_token($email);

            if (!$token) {
                $this->session->set_flashdata('error', 'Email tidak ditemukan.');
                redirect('user/forgot_password');
                return;
            }

            // Send reset email
            if ($this->send_reset_email($email, $token)) {
                $this->session->set_flashdata('success', 'Email reset password telah dikirim. Silakan cek email Anda.');
            } else {
                $this->session->set_flashdata('error', 'Gagal mengirim email reset password. Silakan coba lagi.');
            }

            redirect('user/forgot_password');
        }
    }

    private function send_reset_email($email, $token)
    {
        $this->load->library('email');
        $this->config->load('email', TRUE);
        $email_config = $this->config->item('email');

        $this->email->initialize($email_config);

        $this->email->from($email_config['email_from'], $email_config['email_from_name']);
        $this->email->to($email);
        $this->email->subject('Reset Password - Live Stream');

        $reset_link = site_url('user/reset-password/' . $token);

        $message = "<p>Anda menerima email ini karena ada permintaan reset password untuk akun Anda.</p>";
        $message .= "<p>Silakan klik tautan berikut untuk mereset password Anda:</p>";
        $message .= "<p><a href='{$reset_link}'>{$reset_link}</a></p>";
        $message .= "<p>Jika Anda tidak meminta reset password, abaikan email ini.</p>";

        $this->email->message($message);

        return $this->email->send();
    }

    public function reset_password($token = null)
    {
        if (!$token) {
            show_404();
            return;
        }

        $user = $this->User_model->validate_reset_token($token);

        if (!$user) {
            $this->session->set_flashdata('error', 'Token reset password tidak valid atau sudah kadaluarsa.');
            redirect('user/forgot_password');
            return;
        }

        $data['title'] = 'Reset Password';
        $data['token'] = $token;

        if ($this->input->method() === 'post') {
            $this->form_validation->set_rules('new_password', 'Password Baru', 'required|min_length[6]');
            $this->form_validation->set_rules('confirm_password', 'Konfirmasi Password', 'required|matches[new_password]');

            if ($this->form_validation->run() === FALSE) {
                $this->load->view('templates/header', $data);
                $this->load->view('user/reset_password_form', $data);
                $this->load->view('templates/footer');
            } else {
                $new_password = $this->input->post('new_password');

                if ($this->User_model->update_password_by_token($token, $new_password)) {
                    $this->session->set_flashdata('success', 'Password berhasil diubah. Silakan login dengan password baru Anda.');
                    redirect('login');
                } else {
                    $this->session->set_flashdata('error', 'Gagal mengubah password. Silakan coba lagi.');
                    redirect('user/reset-password/' . $token);
                }
            }
        } else {
            $this->load->view('templates/header', $data);
            $this->load->view('user/reset_password_form', $data);
            $this->load->view('templates/footer');
        }
    }

    public function contact_admin()
    {
        // Check if user is logged in
        if (!$this->session->userdata('logged_in')) {
            redirect('auth/login');
        }

        $data['title'] = 'Kirim Pesan ke Admin';
        $data['user'] = $this->User_model->get_user_by_email($this->session->userdata('email'));

        $this->form_validation->set_rules('subject', 'Subjek', 'required|max_length[100]');
        $this->form_validation->set_rules('message', 'Pesan', 'required|max_length[1000]');

        if ($this->form_validation->run() === FALSE) {
            $this->load->view('templates/header', $data);
            $this->load->view('user/contact_admin', $data);
            $this->load->view('templates/footer');
        } else {
            $subject = $this->input->post('subject');
            $message = $this->input->post('message');
            $user_email = $this->session->userdata('email');
            $user_name = $this->session->userdata('name');

            // Send email to admin
            $this->send_contact_message($user_email, $user_name, $subject, $message);

            $this->session->set_flashdata('success', 'Pesan telah dikirim ke admin. Terima kasih!');
            redirect('user/contact_admin');
        }
    }

    private function send_reset_request($email, $reason)
    {
        $this->load->library('email');

        $this->email->from('noreply@yourdomain.com', 'Live Stream System');
        $this->email->to('admin@yourdomain.com');
        $this->email->subject('Permintaan Reset Password - ' . $email);

        $message = "Permintaan reset password dari user:\n\n";
        $message .= "Email: " . $email . "\n";
        $message .= "Alasan: " . $reason . "\n\n";
        $message .= "Silakan proses permintaan ini sesuai kebijakan.";

        $this->email->message($message);

        if ($this->email->send()) {
            return true;
        } else {
            log_message('error', 'Failed to send reset password request email');
            return false;
        }
    }

    private function send_contact_message($user_email, $user_name, $subject, $message)
    {
        $this->load->library('email');

        $this->email->from($user_email, $user_name);
        $this->email->to('admin@yourdomain.com');
        $this->email->subject('Pesan dari User - ' . $subject);

        $full_message = "Pesan dari user:\n\n";
        $full_message .= "Nama: " . $user_name . "\n";
        $full_message .= "Email: " . $user_email . "\n\n";
        $full_message .= "Subjek: " . $subject . "\n\n";
        $full_message .= "Pesan:\n" . $message;

        $this->email->message($full_message);

        if ($this->email->send()) {
            return true;
        } else {
            log_message('error', 'Failed to send contact message email');
            return false;
        }
    }
}
