<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Settings extends CI_Controller
{

    public function __construct()
    {
        parent::__construct();
        $this->load->model('Settings_model');
        $this->load->helper(['form', 'url', 'file']);
    }

    private function _is_logged_in()
    {
        if (!$this->session->userdata('user_id')) {
            redirect('auth/login');
            return;
        }
    }

    public function theme()
    {
        // Check if user is logged in
        $this->_is_logged_in();

        // Load settings
        $data['settings'] = $this->Settings_model->get_settings();
        if (!$data['settings']) {
            // Create default settings if not exists
            $default_data = [
                'primary_color' => '#4e73df',
                'secondary_color' => '#1cc88a',
                'background_type' => 'gradient',
                'background_value' => 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)',
                'background_overlay' => TRUE,
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s')
            ];
            $this->Settings_model->update_settings($default_data);
            $data['settings'] = $this->Settings_model->get_settings();
        }

        $data['title'] = 'Theme Settings';
        $data['page'] = 'theme';

        // Load views
        $this->load->view('templates/header', $data);
        $this->load->view('settings/theme', $data);
        $this->load->view('templates/footer');
    }

    public function update_theme()
    {
        $this->_is_logged_in();

        $this->load->library('form_validation');

        $this->form_validation->set_rules('primary_color', 'Primary Color', 'required|max_length[7]');
        $this->form_validation->set_rules('secondary_color', 'Secondary Color', 'required|max_length[7]');
        $this->form_validation->set_rules('text_color', 'Text Color', 'required|max_length[7]');
        $this->form_validation->set_rules('background_type', 'Background Type', 'required|in_list[color,gradient,image]');

        if ($this->form_validation->run() === FALSE) {
            $this->theme();
            return;
        }

        $data = [
            'primary_color' => $this->input->post('primary_color'),
            'secondary_color' => $this->input->post('secondary_color'),
            'text_color' => $this->input->post('text_color'),
            'background_type' => $this->input->post('background_type'),
            'background_overlay' => $this->input->post('background_overlay') ? TRUE : FALSE,
            'updated_at' => date('Y-m-d H:i:s')
        ];

        // Handle background value based on type
        switch ($data['background_type']) {
            case 'color':
                $data['background_value'] = $this->input->post('background_color');
                break;
            case 'gradient':
                $data['background_value'] = $this->input->post('background_gradient');
                break;
            case 'image':
                if (!empty($_FILES['background_image']['name'])) {
                    $upload_result = $this->upload_background_image();
                    if ($upload_result['success']) {
                        // Delete old background image if exists
                        $old_settings = $this->Settings_model->get_settings();
                        if ($old_settings && $old_settings->background_type === 'image' && $old_settings->background_value) {
                            $old_path = './uploads/backgrounds/' . $old_settings->background_value;
                            if (file_exists($old_path)) {
                                unlink($old_path);
                            }
                        }
                        $data['background_value'] = $upload_result['file_name'];
                    } else {
                        $this->session->set_flashdata('error', $upload_result['error']);
                        redirect('settings/theme');
                        return;
                    }
                } else {
                    // Keep existing image if no new upload
                    $current_settings = $this->Settings_model->get_settings();
                    $data['background_value'] = $current_settings->background_value ?? '';
                }
                break;
        }

        if ($this->Settings_model->update_settings($data)) {
            $this->session->set_flashdata('success', 'Theme settings updated successfully!');
        } else {
            $this->session->set_flashdata('error', 'Failed to update theme settings.');
        }

        redirect('settings/theme');
    }

    private function upload_background_image()
    {
        $config['upload_path'] = './uploads/backgrounds/';
        $config['allowed_types'] = 'jpg|jpeg|png|gif|webp';
        $config['max_size'] = 2048; // 2MB
        $config['encrypt_name'] = TRUE;

        if (!is_dir($config['upload_path'])) {
            mkdir($config['upload_path'], 0777, TRUE);
        }

        $this->load->library('upload', $config);

        if (!$this->upload->do_upload('background_image')) {
            return [
                'success' => FALSE,
                'error' => $this->upload->display_errors()
            ];
        }

        return [
            'success' => TRUE,
            'file_name' => $this->upload->data('file_name')
        ];
    }

    public function reset_theme()
    {
        $this->_is_logged_in();

        $default_data = [
            'primary_color' => '#4e73df',
            'secondary_color' => '#1cc88a',
            'background_type' => 'gradient',
            'background_value' => 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)',
            'background_overlay' => TRUE,
            'updated_at' => date('Y-m-d H:i:s')
        ];

        if ($this->Settings_model->update_settings($default_data)) {
            $this->session->set_flashdata('success', 'Theme reset to default successfully!');
        } else {
            $this->session->set_flashdata('error', 'Failed to reset theme.');
        }

        redirect('settings/theme');
    }

    // New methods for maintenance mode settings
    public function maintenance()
    {
        $this->_is_logged_in();

        $data['settings'] = $this->Settings_model->get_settings();
        $data['title'] = 'Maintenance Settings';
        $data['page'] = 'maintenance';

        $this->load->view('templates/header', $data);
        $this->load->view('settings/maintenance', $data);
        $this->load->view('templates/footer');
    }

    public function update_maintenance()
    {
        $this->_is_logged_in();

        $this->load->library('form_validation');

        $this->form_validation->set_rules('maintenance_mode', 'Maintenance Mode', 'required|in_list[0,1]');
        $this->form_validation->set_rules('maintenance_message', 'Maintenance Message', 'max_length[255]');

        if ($this->form_validation->run() === FALSE) {
            $this->maintenance();
            return;
        }

        $data = [
            'maintenance_mode' => $this->input->post('maintenance_mode'),
            'maintenance_message' => $this->input->post('maintenance_message'),
            'updated_at' => date('Y-m-d H:i:s')
        ];

        if ($this->Settings_model->update_settings($data)) {
            $this->session->set_flashdata('success', 'Maintenance settings updated successfully!');
        } else {
            $this->session->set_flashdata('error', 'Failed to update maintenance settings.');
        }

        redirect('settings/maintenance');
    }
}
