<?php
defined('BASEPATH') or exit('No direct script access allowed');

/**
 * @property CI_Session $session
 * @property CI_Form_validation $form_validation
 * @property CI_Input $input
 * @property User_model $User_model
 * @property Video_model $Video_model
 */
class Admin extends CI_Controller
{
    public function __construct()
    {
        parent::__construct();
        $this->load->library('VideoService');
        $this->load->model('User_model');
        $this->load->model('Login_history_model');
        $this->load->library('session');
        $this->load->library('form_validation');
        $this->load->helper(['form', 'url', 'theme_helper']);
    }

    public function index()
    {
        redirect('admin/dashboard');
    }

    public function dashboard()
    {
        if (!$this->session->userdata('user_id') || $this->session->userdata('role') !== 'admin') {
            redirect('login');
        }

        $data['total_users'] = $this->User_model->count_users();
        $data['total_videos'] = $this->videoservice->countVideos();
        $data['login_history'] = $this->Login_history_model->get_all_logins();

        $this->load->view('templates/header');
        $this->load->view('templates/sidebar');
        $this->load->view('admin/dashboard', $data);
        $this->load->view('templates/footer');
    }

    public function manage_videos()
    {
        if (!$this->session->userdata('user_id') || $this->session->userdata('role') !== 'admin') {
            redirect('login');
        }

        $data['videos'] = $this->videoservice->getAllVideos();

        $this->load->view('templates/header');
        $this->load->view('templates/sidebar');
        $this->load->view('admin/manage_videos', $data);
        $this->load->view('templates/footer');
    }

    public function edit_video($video_id)
    {
        if (!$this->session->userdata('user_id') || $this->session->userdata('role') !== 'admin') {
            redirect('login');
        }

        $data['video'] = $this->videoservice->getVideo($video_id);

        if (!$data['video']) {
            show_404();
        }

        $this->form_validation->set_rules('title', 'Title', 'required');
        $this->form_validation->set_rules('source', 'Source', 'required');

        if ($this->form_validation->run() === FALSE) {
            $this->load->view('templates/header');
            $this->load->view('templates/sidebar');
            $this->load->view('admin/edit_video', $data);
            $this->load->view('templates/footer');
        } else {
            $update_data = [
                'title' => $this->input->post('title', true),
                'source' => $this->input->post('source', true),
                'video_type' => $this->input->post('video_type', true),
            ];

            // Set source_type based on whether source is a file or link
            $update_data['source_type'] = (strpos($update_data['source'], 'http') === 0) ? 'link' : 'file';

            // Validate Facebook video URL if source contains facebook.com and video type is live
            if ($update_data['video_type'] === 'live' && strpos($update_data['source'], 'facebook.com') !== false) {
                $pattern = '/^https?:\/\/(www\.)?facebook\.com\/[^\/]+\/videos\/[0-9]+\/?/';
                if (!preg_match($pattern, $update_data['source'])) {
                    $this->session->set_flashdata('error', 'URL Facebook tidak valid. Harap masukkan URL video Facebook yang benar.');
                    redirect('admin/edit_video/' . $video_id);
                    return;
                }
            }

            $this->videoservice->updateVideo($video_id, $update_data);
            $this->session->set_flashdata('success', 'Video berhasil diperbarui.');
            redirect('admin/manage_videos');
        }
    }

    public function delete_video($video_id)
    {
        if (!$this->session->userdata('user_id') || $this->session->userdata('role') !== 'admin') {
            redirect('login');
        }

        // Validate video ID before deleting
        $video = $this->videoservice->getVideo($video_id);
        if (!$video) {
            $this->session->set_flashdata('error', 'Video tidak ditemukan atau sudah dihapus.');
            redirect('admin/manage_videos');
            return;
        }

        $this->videoservice->deleteVideo($video_id);
        $this->session->set_flashdata('success', 'Video berhasil dihapus.');
        redirect('admin/manage_videos');
    }

    public function create()
    {
        if (!$this->session->userdata('user_id') || $this->session->userdata('role') !== 'admin') {
            redirect('login');
        }

        $this->form_validation->set_rules('title', 'Title', 'required');
        $this->form_validation->set_rules('embed_link', 'Embed Link', 'required');

        if ($this->form_validation->run() == FALSE) {
            $this->load->view('templates/header');
            $this->load->view('templates/sidebar');
            $this->load->view('video/create');
            $this->load->view('templates/footer');
        } else {
            $data = [
                'user_id' => $this->session->userdata('user_id'),
                'title' => $this->input->post('title'),
                'description' => $this->input->post('description'),
                'embed_link' => $this->input->post('embed_link'),
                'created_by' => $this->session->userdata('user_id'),
            ];

            $this->videoservice->createVideo($data);
            $this->session->set_flashdata('success', 'Video berhasil ditambahkan');
            redirect('admin/manage_videos');
        }
    }

    public function manage_users()
    {
        if (!$this->session->userdata('user_id') || $this->session->userdata('role') !== 'admin') {
            redirect('login');
        }

        $data['users'] = $this->User_model->get_all_users();

        $this->load->view('templates/header');
        $this->load->view('templates/sidebar');
        $this->load->view('admin/users', $data);
        $this->load->view('templates/footer');
    }

    public function login_history()
    {
        if (!$this->session->userdata('user_id') || $this->session->userdata('role') !== 'admin') {
            redirect('login');
        }

        $data['login_history'] = $this->Login_history_model->get_all_logins();

        $this->load->view('templates/header');
        $this->load->view('templates/sidebar');
        $this->load->view('admin/login_history', $data);
        $this->load->view('templates/footer');
    }

    public function edit_user($user_id)
    {
        if (!$this->session->userdata('user_id') || $this->session->userdata('role') !== 'admin') {
            redirect('login');
        }

        $data['user'] = $this->User_model->get_user_by_id($user_id);

        if (!$data['user']) {
            show_404();
        }

        $this->form_validation->set_rules('username', 'Username', 'required');
        $this->form_validation->set_rules('email', 'Email', 'required|valid_email');
        $this->form_validation->set_rules('role', 'Role', 'required|in_list[user,admin]');
        $this->form_validation->set_rules('password', 'Password', 'min_length[6]');

        if ($this->form_validation->run() === FALSE) {
            $this->load->view('templates/header');
            $this->load->view('templates/sidebar');
            $this->load->view('admin/edit_user', $data);
            $this->load->view('templates/footer');
        } else {
            $update_data = [
                'username' => $this->input->post('username', true),
                'email' => $this->input->post('email', true),
                'role' => $this->input->post('role', true),
            ];

            $password = $this->input->post('password', true);
            if (!empty($password)) {
                $update_data['password'] = password_hash($password, PASSWORD_DEFAULT);
            }

            if (!empty($_FILES['profile_photo']['name'])) {
                $config['upload_path'] = './assets/sbadmin2/uploads/';
                $config['allowed_types'] = 'jpg|jpeg|png|gif';
                $config['max_size'] = 2048; // 2MB
                $config['file_name'] = time() . '_' . $_FILES['profile_photo']['name'];
                $config['file_ext_tolower'] = TRUE;

                $this->load->library('upload', $config);

                if (!$this->upload->do_upload('profile_photo')) {
                    $error = $this->upload->display_errors('<div class="alert alert-danger">', '</div>');
                    $this->session->set_flashdata('error', $error);
                    redirect('admin/edit_user/' . $user_id);
                    return;
                } else {
                    $upload_data = $this->upload->data();
                    $update_data['profile_photo'] = $upload_data['file_name'];
                }
            }

            $this->User_model->update_user($user_id, $update_data);
            $this->session->set_flashdata('success', 'User berhasil diperbarui.');
            redirect('admin/manage_users');
        }
    }

    public function delete_user($user_id)
    {
        if (!$this->session->userdata('user_id') || $this->session->userdata('role') !== 'admin') {
            redirect('login');
        }

        $this->User_model->delete_user($user_id);
        $this->session->set_flashdata('success', 'User berhasil dihapus.');
        redirect('admin/manage_users');
    }

    public function add_user()
    {
        if (!$this->session->userdata('user_id') || $this->session->userdata('role') !== 'admin') {
            redirect('auth/login');
        }

        $this->form_validation->set_rules('username', 'Username', 'required|min_length[3]|max_length[50]|is_unique[users.username]');
        $this->form_validation->set_rules('email', 'Email', 'required|valid_email|is_unique[users.email]');
        $this->form_validation->set_rules('password', 'Password', 'required|min_length[6]');
        $this->form_validation->set_rules('role', 'Role', 'required|in_list[user,admin]');

        if ($this->form_validation->run() === FALSE) {
            $this->load->view('templates/header');
            $this->load->view('templates/sidebar');
            $this->load->view('admin/add_user');
            $this->load->view('templates/footer');
        } else {
            $email = $this->input->post('email', true);
            $password = password_hash($this->input->post('password', true), PASSWORD_DEFAULT);
            $role = $this->input->post('role', true);

            $profile_photo = null;
            if (!empty($_FILES['profile_photo']['name'])) {
                $config['upload_path'] = './assets/sbadmin2/uploads/';
                $config['allowed_types'] = 'jpg|jpeg|png|gif';
                $config['max_size'] = 2048; // 2MB
                $config['file_name'] = time() . '_' . $_FILES['profile_photo']['name'];
                $config['file_ext_tolower'] = TRUE;

                $this->load->library('upload', $config);

                if (!$this->upload->do_upload('profile_photo')) {
                    $error = $this->upload->display_errors('<div class="alert alert-danger">', '</div>');
                    $this->session->set_flashdata('error', $error);
                    redirect('admin/add_user');
                    return;
                } else {
                    $upload_data = $this->upload->data();
                    $profile_photo = $upload_data['file_name'];
                }
            }

            $data = [
                'email' => $email,
                'password' => $password,
                'role' => $role,
                'profile_photo' => $profile_photo,
                'created_at' => date('Y-m-d H:i:s'),
            ];

            $this->User_model->insert_user($data);
            $this->session->set_flashdata('success', 'User successfully added.');
            redirect('admin/add_user');
        }
    }

    public function upload_video()
    {
        // Cek login dan role
        if (!$this->session->userdata('user_id') || $this->session->userdata('role') !== 'admin') {
            redirect('auth/login');
        }

        // Validasi form input
        $this->form_validation->set_rules('title', 'Title', 'required');
        $this->form_validation->set_rules('description', 'Description', 'required');

        if ($this->form_validation->run() === FALSE) {
            $this->load->view('templates/header');
            $this->load->view('templates/sidebar');
            $this->load->view('admin/upload_video');
            $this->load->view('templates/footer');
            return;
        }

        $title = $this->input->post('title', TRUE);
        $description = $this->input->post('description', TRUE);
        $video_type = $this->input->post('video_type', TRUE);
        $video_link = $this->input->post('video_link', TRUE);
        $video_path = null;
        $thumbnail_path = null;

        // Load upload library once without config
        $this->load->library('upload');

        // Cek apakah file video diupload
        if (!empty($_FILES['video_file']['name'])) {
            $config['upload_path'] = './assets/sbadmin2/uploads/';
            $config['allowed_types'] = 'mp4|avi|mov|wmv|flv|webm|mkv';
            $config['max_size'] = 500000; // 500MB
            $config['file_name'] = time() . '_' . $_FILES['video_file']['name'];
            $config['file_ext_tolower'] = TRUE;

            $this->upload->initialize($config);

            if (!$this->upload->do_upload('video_file')) {
                $error = $this->upload->display_errors('<div class="alert alert-danger">', '</div>');
                $this->session->set_flashdata('error', $error);
                redirect('admin/upload_video');
                return;
            }

            $upload_data = $this->upload->data();
            $video_path = 'assets/sbadmin2/uploads/' . $upload_data['file_name'];
        }

        // Cek apakah file thumbnail diupload
        if (!empty($_FILES['thumbnail_file']['name'])) {
            $config_thumb['upload_path'] = './assets/sbadmin2/uploads/';
            $config_thumb['allowed_types'] = 'jpg|jpeg|png|gif';
            $config_thumb['max_size'] = 5000; // 5MB
            $config_thumb['file_name'] = 'thumb_' . time() . '_' . $_FILES['thumbnail_file']['name'];
            $config_thumb['file_ext_tolower'] = TRUE;

            $this->upload->initialize($config_thumb);

            if (!$this->upload->do_upload('thumbnail_file')) {
                $error = $this->upload->display_errors('<div class="alert alert-danger">', '</div>');
                $this->session->set_flashdata('error', $error);
                redirect('admin/upload_video');
                return;
            }

            $upload_data_thumb = $this->upload->data();
            $thumbnail_path = 'assets/sbadmin2/uploads/' . $upload_data_thumb['file_name'];
        }

        // Validasi: salah satu harus diisi
        if (empty($video_path) && empty($video_link)) {
            $this->session->set_flashdata('error', '<div class="alert alert-danger">Please upload a video file or provide a video link.</div>');
            redirect('admin/upload_video');
            return;
        }

        // Additional validation for Telegram links
        if (!empty($video_link) && (strpos($video_link, 'telegram.org') !== false || strpos($video_link, 't.me') !== false)) {
            if (!preg_match('/\.(mp4|webm|ogg)$/i', $video_link)) {
                $this->session->set_flashdata('error', '<div class="alert alert-danger">Telegram video links must be direct video file URLs ending with .mp4, .webm, or .ogg.</div>');
                redirect('admin/upload_video');
                return;
            }
        }

        // Validasi khusus untuk video live
        if ($video_type === 'live') {
            // Hanya boleh menggunakan link video (tidak boleh upload file)
            if (!empty($video_path)) {
                $this->session->set_flashdata('error', '<div class="alert alert-danger">Video live hanya boleh menggunakan link, tidak bisa upload file.</div>');
                redirect('admin/upload_video');
                return;
            }

            // Harus merupakan link Facebook Live
            if (empty($video_link) || strpos($video_link, 'facebook.com') === false) {
                $this->session->set_flashdata('error', '<div class="alert alert-danger">Video live harus menggunakan link Facebook Live.</div>');
                redirect('admin/upload_video');
                return;
            }

            // Validasi format link Facebook Live
            $pattern = '/^https?:\/\/(www\.)?facebook\.com\/[^\/]+\/videos\/[0-9]+\/?/';
            if (!preg_match($pattern, $video_link)) {
                $this->session->set_flashdata('error', '<div class="alert alert-danger">URL Facebook Live tidak valid. Harap masukkan URL video Facebook yang benar.</div>');
                redirect('admin/upload_video');
                return;
            }
        }

        // Additional validation for Telegram links
        if (!empty($video_link) && (strpos($video_link, 'telegram.org') !== false || strpos($video_link, 't.me') !== false)) {
            if (!preg_match('/\.(mp4|webm|ogg)$/i', $video_link)) {
                $this->session->set_flashdata('error', '<div class="alert alert-danger">Telegram video links must be direct video file URLs ending with .mp4, .webm, or .ogg.</div>');
                redirect('admin/upload_video');
                return;
            }
        }

        // Siapkan data untuk dimasukkan ke database
        $data = [
            'user_id' => $this->session->userdata('user_id'),
            'title' => $title,
            'description' => $description,
            'video_type' => $video_type,
            'source_type' => !empty($video_path) ? 'file' : 'link',
            'source' => !empty($video_path) ? $video_path : $video_link,
            'thumbnail' => $thumbnail_path,
            'created_at' => date('Y-m-d H:i:s'),
        ];

        $video_id = $this->Video_model->upload_video($data);
        $this->session->set_flashdata('success', '✅ Video uploaded successfully. ID Video: ' . $video_id);
        redirect('admin/manage_videos');
    }

}
